implementation module Highscore


/*	General utility for reading/writing high scores to Files and displaying current high scores.
	This module uses the Object I/O library, version 1.1
*/

import	StdEnv, StdIO

::	HiScores
	:== [HiScore]
::	HiScore
	=	{	name	:: !String
		,	score	:: !Int
		}

//	Read in the high scores:
readHiScores :: !String !*Files -> (!(!*File,!HiScores),!*Files)
readHiScores fname files
	#	(exists,file,files)	= fopen fpath FReadData files
	|	exists
		#	(highs,file)		= readHighs file
		=	((file,highs),files)
	|	otherwise
		#	(_,create,files)	= fopen fpath FWriteData files
		=	((create,[]),files)
where
	fpath = homepath fname
	
	readHighs :: !*File -> (!HiScores,!*File)
	readHighs file
		|	sfend file
			=	([],file)
		#	(name, file) = freads file 13
		#	(ok,hi,file) = freadi file
		|	not ok
			=	([],file)
		#	(ok,_, file) = freadc file
		|	not ok
			=	([],file)
		#	(rest, file) = readHighs file
		|	otherwise
			=	([{name=name,score=hi}:rest],file)

//	Write the high scores:
writeHiScores :: !*File !HiScores !*Files -> *Files
writeHiScores file highs files
	#	(ok,file)	= freopen file FWriteData
	|	not ok
		=	abort "Could not reopen file.\n"
	#	(_,files)	= fclose (file<<<highs) files
	|	otherwise
		=	files

instance <<< HiScore where
	(<<<) :: !*File !HiScore -> *File
	(<<<) f {name,score}
		=	f<<<take13 name<<<score<<<'\n'
	where
		take13 :: !String -> String
		take13 string = (string+++"             ")%(0,12)

instance <<< [x] | <<< x where
	(<<<) :: !*File ![x] -> *File | <<< x
	(<<<) f [x:xs]	= f<<<x<<<xs
	(<<<) f _		= f


//	Determine whether, given the number of high scores, a given score is actually a new high score:
itsAHighScore :: !Int !Int !HiScores -> Bool
itsAHighScore nrOfHiScores score scores
	|	score==0
		=	False
	|	length scores<nrOfHiScores
		=	True
	|	otherwise
		=	isItReallyAHighScore score scores
where
	isItReallyAHighScore :: !Int !HiScores -> Bool
	isItReallyAHighScore score` [{score}:hiscores]
		=	score`>score || isItReallyAHighScore score` hiscores
	isItReallyAHighScore _ _
		=	False


//	Add a HiScore to the current list of high scores:
addScore :: !Int !HiScore !HiScores -> HiScores
addScore nrOfHighScores hi hiscores
	=	take nrOfHighScores (addscore hi hiscores)
where
	addscore :: !HiScore !HiScores -> HiScores
	addscore hi` hiscores=:[hi:his]
		|	hi.score>hi`.score
			=	[hi  : addscore hi` his]
		|	otherwise
			=	[hi` : hiscores]
	addscore hi` _
		=	[hi`]


//	Display high scores in a modal dialog to the user:
showHiScores :: Id String !HiScores !(PSt .l .p) -> PSt .l .p
showHiScores id header highs pState
	#	(okId,pState)	= accPIO openId pState
	#	(_,pState)		= openModalDialog undef (dialog okId) pState
	=	pState
where
	dialog okId
		= Dialog "High Scores"
			(	TextControl header	[ControlPos (Center,zero)]
			:+:	text
			:+:	ButtonControl "OK"	[ControlId okId,ControlPos (Center,zero),ControlFunction (noLS closeActiveWindow)]
			)
			[	WindowId	id
			,	WindowOk	okId
			]
	text= if (isEmpty highs)
			(ListLS [TextControl "No high scores available." []])
			(ListLS [TextControl (toString hi+++". "+++name+++" "+++toString score) [] \\ (hi,{name,score}) <- zip2 [1..] highs])
	closeActiveWindow pState
		#	(Just wid,pState)	= accPIO getActiveWindow pState
		=	closeWindow wid pState
